// ts3-viewer.js
(function($){
    'use strict';

    const cfg = window.ts3viewer_cfg || {};
    const AJAX_URL = cfg.ajaxurl || (window.ajaxurl ? window.ajaxurl : '/wp-admin/admin-ajax.php');
    const NONCE = cfg.nonce || '';
    const INTERVAL_MS = parseInt(cfg.interval_ms || 10000, 10);
    const CHARSET = 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789!@#$%^&*()_+-=[]{};:,.<>?';

    function animateNicknames() {
        document.querySelectorAll('.ts3-client .scramble').forEach((el, i) => {
            setTimeout(() => scrambleText(el), i * 80);
        });
    }

    function scrambleText(element) {
        const original = element.getAttribute('data-text') || element.textContent || '';
        if (!original) return;
        let iter = 0;
        const MAX_ITERS = 60;
        const SPEED = 20;

        const handle = setInterval(() => {
            const out = original.split('').map((ch, idx) => {
                if (idx < iter / 2) return original[idx];
                return CHARSET[Math.floor(Math.random() * CHARSET.length)];
            }).join('');
            element.textContent = out;
            iter++;
            if (iter >= MAX_ITERS) {
                clearInterval(handle);
                element.textContent = original;
            }
        }, SPEED);
    }

    function fadeInClients() {
        const clients = document.querySelectorAll('.ts3-client');
        clients.forEach((el, i) => {
            el.classList.remove('ts3-fade-in');
            setTimeout(() => el.classList.add('ts3-fade-in'), 80 + i * 40);
        });
    }

    function animateFlags() {
        const flags = document.querySelectorAll('.ts3-country');
        flags.forEach((flag, i) => {
            flag.style.animation = 'none';
            flag.style.transition = 'transform 0.3s cubic-bezier(0.4, 0, 0.2, 1)';
            void flag.offsetWidth;

            const delay = (i % 8) * 0.15;
            flag.style.animation = `flagFloat 4s ease-in-out infinite ${delay}s`;

            const enter = () => flag.style.animation = 'flagWave 0.8s ease-in-out';
            const leave = () => {
                const newDelay = (Array.from(flags).indexOf(flag) % 8) * 0.15;
                flag.style.animation = `flagFloat 4s ease-in-out infinite ${newDelay}s`;
            };

            if (flag._flagHandlers) {
                flag.removeEventListener('mouseenter', flag._flagHandlers.enter);
                flag.removeEventListener('mouseleave', flag._flagHandlers.leave);
            }

            flag.addEventListener('mouseenter', enter);
            flag.addEventListener('mouseleave', leave);
            flag._flagHandlers = { enter, leave };
        });
    }

    function refreshViewer() {
        const container = $('#ts3viewer-content');
        if (!container.length) return;

        $.ajax({
            url: AJAX_URL,
            method: 'POST',
            data: { action: 'ts3viewer_refresh', nonce: NONCE },
            dataType: 'html',
            timeout: 10000
        })
        .done(function(responseHtml){
            const wasEmpty = container.html().trim() === '';
            container.html(responseHtml);
            fadeInClients();
            animateNicknames();
            setTimeout(animateFlags, 200); // Просто запускаем анимацию

            try {
                const evt = new CustomEvent('ts3viewerRefresh', { detail: { time: Date.now(), wasEmpty } });
                document.dispatchEvent(evt);
            } catch { $(document).trigger('ts3viewerRefresh', [{ wasEmpty }]); }
        })
        .fail(function(jqXHR, textStatus, errorThrown){
            console.error('TS3 Viewer refresh failed:', textStatus, errorThrown);
        });
    }

    $(document).ready(function(){
        const init = () => {
            fadeInClients();
            animateNicknames();
            animateFlags();
        };

        setTimeout(() => {
            if (document.fonts) document.fonts.ready.then(init);
            else init();
        }, 120);

        if (INTERVAL_MS > 0) setInterval(refreshViewer, INTERVAL_MS);

        let resizeTimer;
        $(window).on('resize', () => {
            clearTimeout(resizeTimer);
            resizeTimer = setTimeout(animateFlags, 200);
        });

        const observer = new MutationObserver(() => {
            setTimeout(animateFlags, 200);
        });
        const target = document.querySelector('#ts3viewer-content');
        if (target) observer.observe(target, { childList: true, subtree: true });
    });

})(jQuery);