<?php
/*
Plugin Name: TS3 Viewer (Animated with Glassmorphism)
Description: Стильный просмотрщик сервера TeamSpeak 3 с автообновлением, счётчиком пользователей, флагами стран по IP.
Version: 3.8.1
Author: dMITRI4_26
*/
if (!defined('ABSPATH')) exit;
// Добавляем страницу настроек WP → Settings → TS3 Viewer
function ts3viewer_register_settings() {
    add_option('ts3_host', '127.0.0.1');
    add_option('ts3_query_port', 10011);
    add_option('ts3_server_port', 9987);
    add_option('ts3_query_user', '');
    add_option('ts3_query_pass', '');
    add_option('ts3_enable_geolocation', true); // Опция для отключения геолокации
    add_option('ts3_cache_ttl', 15);
    register_setting('ts3viewer_options_group', 'ts3_host', 'sanitize_text_field');
    register_setting('ts3viewer_options_group', 'ts3_query_port', 'intval');
    register_setting('ts3viewer_options_group', 'ts3_server_port', 'intval');
    register_setting('ts3viewer_options_group', 'ts3_query_user', 'sanitize_text_field');
    register_setting('ts3viewer_options_group', 'ts3_query_pass', 'sanitize_text_field'); // Для пароля используем sanitize_text_field, но в реальности лучше шифровать
    register_setting('ts3viewer_options_group', 'ts3_enable_geolocation', 'boolval');
    register_setting('ts3viewer_options_group', 'ts3_cache_ttl', 'intval');
}
add_action('admin_init', 'ts3viewer_register_settings');
function ts3viewer_add_menu() {
    add_options_page('TS3 Viewer Settings', 'TS3 Viewer', 'manage_options', 'ts3viewer', 'ts3viewer_options_page');
}
add_action('admin_menu', 'ts3viewer_add_menu');
function ts3viewer_options_page() {
    if (!current_user_can('manage_options')) {
        wp_die(__('You do not have sufficient permissions to access this page.'));
    }
    ?>
    <div class="wrap">
        <h2>TS3 Viewer Settings</h2>
        <form method="post" action="options.php">
            <?php settings_fields('ts3viewer_options_group'); ?>
            <table class="form-table">
                <tr valign="top">
                    <th scope="row">TS3 Host</th>
                    <td><input type="text" name="ts3_host" value="<?php echo esc_attr(get_option('ts3_host')); ?>" /></td>
                </tr>
                <tr valign="top">
                    <th scope="row">TS3 Query Port</th>
                    <td><input type="number" name="ts3_query_port" value="<?php echo esc_attr(get_option('ts3_query_port')); ?>" /></td>
                </tr>
                <tr valign="top">
                    <th scope="row">TS3 Server Port</th>
                    <td><input type="number" name="ts3_server_port" value="<?php echo esc_attr(get_option('ts3_server_port')); ?>" /></td>
                </tr>
                <tr valign="top">
                    <th scope="row">TS3 Query User</th>
                    <td><input type="text" name="ts3_query_user" value="<?php echo esc_attr(get_option('ts3_query_user')); ?>" /></td>
                </tr>
                <tr valign="top">
                    <th scope="row">TS3 Query Password</th>
                    <td><input type="password" name="ts3_query_pass" value="<?php echo esc_attr(get_option('ts3_query_pass')); ?>" /></td>
                </tr>
                <tr valign="top">
                    <th scope="row">Enable Geolocation (by IP)</th>
                    <td><input type="checkbox" name="ts3_enable_geolocation" value="1" <?php checked(1, get_option('ts3_enable_geolocation')); ?> /></td>
                </tr>
                <tr valign="top">
                    <th scope="row">Cache TTL (seconds)</th>
                    <td><input type="number" name="ts3_cache_ttl" value="<?php echo esc_attr(get_option('ts3_cache_ttl')); ?>" /></td>
                </tr>
            </table>
            <?php submit_button(); ?>
        </form>
    </div>
    <?php
}
/**
 * Получение данных с сервера TS3 через ServerQuery
 */
function ts3viewer_get_data_html() {
    $cache_key = 'ts3viewer_data';
    $cached = get_transient($cache_key);
    if ($cached !== false) return $cached;
    $ts3_host = get_option('ts3_host', '127.0.0.1');
    $ts3_query_port = get_option('ts3_query_port', 10011);
    $ts3_server_port = get_option('ts3_server_port', 9987);
    $ts3_query_user = get_option('ts3_query_user', '');
    $ts3_query_pass = get_option('ts3_query_pass', '');
    $ts3_enable_geolocation = get_option('ts3_enable_geolocation', true);
    $ts3_cache_ttl = get_option('ts3_cache_ttl', 15);
    if (empty($ts3_query_pass)) {
        return "<div class='ts3-error'>⚠️ Настройте пароль в настройках плагина</div>";
    }
    try {
        ini_set('max_execution_time', 10);
        // Подключение к ServerQuery
        $fp = @fsockopen($ts3_host, $ts3_query_port, $errno, $errstr, 5);
        if (!$fp) {
            throw new Exception("Не удалось подключиться: $errstr ($errno)");
        }
        // Пропускаем приветственное сообщение с валидацией
        stream_set_timeout($fp, 5);
        $welcome = '';
        while (!feof($fp)) {
            $line = fgets($fp);
            $welcome .= $line;
            if (strpos($line, 'TS3') !== false) break;
        }
        if (strpos($welcome, 'TS3') === false) {
            fclose($fp);
            throw new Exception("Неверный ответ приветствия от сервера");
        }
        // Экранирование пароля
        $escaped_pass = str_replace(' ', '\s', $ts3_query_pass);
        // Авторизация
        fwrite($fp, "login " . $ts3_query_user . " " . $escaped_pass . "\n");
        $response = '';
        while (!feof($fp)) {
            $line = fgets($fp);
            $response .= $line;
            if (strpos($line, 'error id=') !== false) break;
        }
        // Валидация ответа
        if (strlen($response) > 500 || strpos($response, 'error id=0') === false) { // Ограничение длины и проверка успеха
            fclose($fp);
            throw new Exception("Ошибка авторизации или подозрительный ответ: " . esc_html(substr($response, 0, 100)));
        }
        // Выбор виртуального сервера
        fwrite($fp, "use port=" . $ts3_server_port . "\n");
        $response = '';
        while (!feof($fp)) {
            $line = fgets($fp);
            $response .= $line;
            if (strpos($line, 'error id=') !== false) break;
        }
        if (strpos($response, 'error id=0') === false) {
            fclose($fp);
            throw new Exception("Ошибка выбора сервера: " . esc_html($response));
        }
        // Получение информации о сервере
        fwrite($fp, "serverinfo\n");
        $server_info = '';
        while (!feof($fp)) {
            $line = fgets($fp);
            $server_info .= $line;
            if (strpos($line, 'error id=') !== false) break;
        }
        if (strlen($server_info) > 5000) { // Увеличено
            fclose($fp);
            throw new Exception("Слишком большой ответ serverinfo");
        }
        preg_match('/virtualserver_name=([^ ]+)/', $server_info, $server_name_match);
        preg_match('/virtualserver_maxclients=([0-9]+)/', $server_info, $max_clients_match);
        $server_name = isset($server_name_match[1]) ? str_replace('\s', ' ', $server_name_match[1]) : 'TeamSpeak Server';
        $max_clients = isset($max_clients_match[1]) ? (int)$max_clients_match[1] : 0;
        // Получение списка клиентов
        fwrite($fp, "clientlist -away -voice -times -info\n");
        $client_data = '';
        while (!feof($fp)) {
            $line = fgets($fp);
            $client_data .= $line;
            if (strpos($line, 'error id=') !== false) break;
        }
        if (strlen($client_data) > 20000) { // Увеличено
            fclose($fp);
            throw new Exception("Слишком большой список клиентов");
        }
        $clients = explode('|', trim($client_data));
        $client_list = [];
        $total_clients = 0;
        $has_country = false;
        foreach ($clients as $client) {
            preg_match('/clid=([0-9]+)/', $client, $clid_match);
            preg_match('/cid=([0-9]+)/', $client, $cid_match);
            preg_match('/client_nickname=([^ ]+)/', $client, $nickname_match);
            preg_match('/client_type=([0-1])/', $client, $type_match);
            preg_match('/client_away=([0-1])/', $client, $away_match);
            preg_match('/client_input_muted=([0-1])/', $client, $input_muted_match);
            preg_match('/client_output_muted=([0-1])/', $client, $output_muted_match);
            preg_match('/client_lastconnected=([0-9]+)/', $client, $lastconnected_match);
            if (isset($clid_match[1]) && isset($cid_match[1]) && isset($nickname_match[1]) && isset($type_match[1])) {
                if ($type_match[1] == 0) { // Учитываем только обычных пользователей
                    $total_clients++;
                    // Получение IP-адреса, client_country и input_hardware через clientinfo
                    $ip_address = '';
                    $country_code = '';
                    $input_hardware = 0;
                    fwrite($fp, "clientinfo clid={$clid_match[1]}\n");
                    $client_info = '';
                    while (!feof($fp)) {
                        $line = fgets($fp);
                        $client_info .= $line;
                        if (strpos($line, 'error id=') !== false) break;
                    }
                    if (strlen($client_info) > 5000) { // Увеличено
                        fclose($fp);
                        throw new Exception("Слишком большой ответ clientinfo");
                    }
                    preg_match('/connection_client_ip=([^ ]+)/', $client_info, $ip_match);
                    preg_match('/client_country=([A-Z]{2})/', $client_info, $country_match);
                    preg_match('/client_input_hardware=([0-1])/', $client_info, $hardware_match);
                    $ip_address = isset($ip_match[1]) ? trim($ip_match[1]) : '';
                    $country_code = isset($country_match[1]) ? strtoupper($country_match[1]) : '';
                    $input_hardware = isset($hardware_match[1]) ? (int)$hardware_match[1] : 0;
                    // Определение страны по IP, если client_country пустой и геолокация включена
                    if (empty($country_code) && $ts3_enable_geolocation && !empty($ip_address) && filter_var($ip_address, FILTER_VALIDATE_IP)) {
                        $query = http_build_query(['fields' => 'countryCode']);
                        $api_url = "https://ip-api.com/json/" . urlencode($ip_address) . "?" . $query;
                        $response = wp_remote_get($api_url, ['timeout' => 2]);
                        if (!is_wp_error($response) && wp_remote_retrieve_response_code($response) === 200) {
                            $data = json_decode(wp_remote_retrieve_body($response), true);
                            if (isset($data['countryCode']) && !empty($data['countryCode'])) {
                                $country_code = strtoupper($data['countryCode']);
                                $has_country = true;
                            }
                        } else {
                            // Резервный API
                            $api_url = "https://ipwhois.app/json/" . urlencode($ip_address);
                            $response = wp_remote_get($api_url, ['timeout' => 2]);
                            if (!is_wp_error($response) && wp_remote_retrieve_response_code($response) === 200) {
                                $data = json_decode(wp_remote_retrieve_body($response), true);
                                if (isset($data['country_code']) && !empty($data['country_code'])) {
                                    $country_code = strtoupper($data['country_code']);
                                    $has_country = true;
                                }
                            }
                        }
                    }
                    $client_list[$cid_match[1]][] = [
                        'clid' => $clid_match[1],
                        'nickname' => str_replace('\s', ' ', $nickname_match[1]),
                        'away' => isset($away_match[1]) ? (int)$away_match[1] : 0,
                        'input_muted' => isset($input_muted_match[1]) ? (int)$input_muted_match[1] : 0,
                        'output_muted' => isset($output_muted_match[1]) ? (int)$output_muted_match[1] : 0,
                        'lastconnected' => isset($lastconnected_match[1]) ? (int)$lastconnected_match[1] : 0,
                        'country' => $country_code,
                        'input_hardware' => $input_hardware,
                    ];
                }
            }
        }
        // Получение списка каналов
        fwrite($fp, "channellist -flags\n");
        $channel_data = '';
        while (!feof($fp)) {
            $line = fgets($fp);
            $channel_data .= $line;
            if (strpos($line, 'error id=') !== false) break;
        }
        if (strlen($channel_data) > 10000) { // Увеличено
            fclose($fp);
            throw new Exception("Слишком большой список каналов");
        }
        $channels = explode('|', trim($channel_data));
        $channel_list = [];
        foreach ($channels as $channel) {
            preg_match('/cid=([0-9]+)/', $channel, $cid_match);
            preg_match('/channel_name=([^ ]+)/', $channel, $name_match);
            preg_match('/channel_flag_default=([0-1])/', $channel, $flag_default_match);
            preg_match('/channel_flag_permanent=([0-1])/', $channel, $flag_permanent_match);
            if (isset($cid_match[1]) && isset($name_match[1])) {
                $channel_list[$cid_match[1]] = [
                    'name' => str_replace('\s', ' ', $name_match[1]),
                    'is_spacer' => (strpos($name_match[1], '[*spacer') !== false || strpos($name_match[1], '[spacer') !== false),
                    'is_default' => isset($flag_default_match[1]) ? (int)$flag_default_match[1] : 0,
                    'is_permanent' => isset($flag_permanent_match[1]) ? (int)$flag_permanent_match[1] : 0,
                ];
            }
        }
        fclose($fp);
        // Формирование HTML
        $html = "<div class='ts3-container'>";
        $html .= "<div class='ts3-header'>🖧 <span class='scramble' data-text=\"".esc_attr($server_name)."\">".esc_html($server_name)."</span></div>";
        $html .= "<div class='ts3-subheader'>Пользователи: {$total_clients} / {$max_clients}</div>";
        if (!$has_country && $total_clients > 0 && $ts3_enable_geolocation) {
            $html .= "<div class='ts3-warning'></div>"; // Можно задать ошибку >⚠️ Страна не определяется<
        }
        $html .= "<ul class='ts3-channels'>";
        $count = 0;
        foreach ($channel_list as $cid => $channel) {
            if (!isset($client_list[$cid]) || empty($client_list[$cid])) continue;
            $channel_icon = '📁';
            if ($channel['is_spacer']) {
                $channel_icon = '➖';
            } elseif ($channel['is_default']) {
                $channel_icon = '🏠';
            } elseif (stripos($channel['name'], 'music') !== false || stripos($channel['name'], 'музыка') !== false) {
                $channel_icon = '🎵';
            }
            $html .= "<li class='ts3-channel'><span class='ts3-icon'>{$channel_icon}</span> " . esc_html($channel['name']) . "<ul>";
            foreach ($client_list[$cid] as $client) {
                $status_icon = '';
                if ($client['away'] === 1) {
                    $status_icon = '<img src="' . set_url_scheme(plugin_dir_url(__FILE__), 'https') . 'icons/pack1/away.png" alt="Away" class="ts3-status-icon">';
                } elseif ($client['output_muted'] === 1) {
                    $status_icon = '<img src="' . set_url_scheme(plugin_dir_url(__FILE__), 'https') . 'icons/pack1/sound_muted.png" alt="Sound Off" class="ts3-status-icon">';
                } elseif ($client['input_hardware'] === 0) {
                    $status_icon = '<img src="' . set_url_scheme(plugin_dir_url(__FILE__), 'https') . 'icons/pack1/micro_hardware_off.png" alt="Mic Hardware Off" class="ts3-status-icon">';
                } elseif ($client['input_muted'] === 1) {
                    $status_icon = '<img src="' . set_url_scheme(plugin_dir_url(__FILE__), 'https') . 'icons/pack1/micro_off.png" alt="Mic Off" class="ts3-status-icon">';
                } else {
                    $status_icon = '<img src="' . set_url_scheme(plugin_dir_url(__FILE__), 'https') . 'icons/pack1/online.png" alt="Online" class="ts3-status-icon">';
                }
                $country_flag = '';
                if (!empty($client['country'])) {
                    $country_code = strtoupper($client['country']);
                    if (strlen($country_code) === 2 && ctype_alpha($country_code)) {
                        $country_flag = mb_convert_encoding('&#' . (127397 + ord($country_code[0])) . ';&#' . (127397 + ord($country_code[1])) . ';', 'UTF-8', 'HTML-ENTITIES');
                    }
                }
                $connected_time = $client['lastconnected'] > 0 ? (time() - $client['lastconnected']) : 0;
                $hours = floor($connected_time / 3600);
                $minutes = floor(($connected_time % 3600) / 60);
                $online_text = ($hours > 0) ? "В онлайне {$hours} ч {$minutes} мин" : "В онлайне {$minutes} мин";
                $html .= "<li class='ts3-client' title='" . esc_attr($online_text) . "'><span class='ts3-icon'>{$status_icon}</span><span class='scramble' data-text=\"".esc_attr($client['nickname'])."\">".esc_html($client['nickname'])."</span><span class='ts3-country'>{$country_flag}</span></li>";
            }
            $html .= "</ul></li>";
            if (++$count > 200) break;
        }
        if ($count == 0) {
            $html .= "<div class='ts3-empty'>ℹ️ Нет активных пользователей</div>";
        }
        $html .= "</ul></div>";
        set_transient($cache_key, $html, $ts3_cache_ttl);
        return $html;
    } catch (Exception $e) {
        error_log("TS3 Viewer Error: " . $e->getMessage()); // Логирование в WP debug.log
        return "<div class='ts3-error'>⚠️ Ошибка соединения с сервером</div>";
    }
}


/* === ПОДКЛЮЧЕНИЕ CSS + JS === */
function ts3viewer_enqueue_assets() {
    // Подключаем стили
    wp_enqueue_style('ts3viewer-style', plugin_dir_url(__FILE__) . 'ts3-viewer.css', [], '3.8.0');

    // Подключаем JS
    wp_enqueue_script('ts3viewer-script', plugin_dir_url(__FILE__) . 'ts3-viewer.js', ['jquery'], '3.8.0', true);

    // Передаём AJAX-параметры в JavaScript
    wp_localize_script('ts3viewer-script', 'ts3viewer_cfg', [
        'ajaxurl' => admin_url('admin-ajax.php'),           // URL для AJAX
        'nonce'   => wp_create_nonce('ts3viewer_refresh'),  // безопасность
        'interval_ms' => 30000                              // интервал обновления (30 сек)
    ]);
}
add_action('wp_enqueue_scripts', 'ts3viewer_enqueue_assets');


/* === AJAX-ОБНОВЛЕНИЕ (основная логика обновления контента) === */
function ts3viewer_ajax_refresh() {
    // Проверяем nonce для безопасности
    check_ajax_referer('ts3viewer_refresh', 'nonce');

    // Получаем новый HTML контент из основной функции
    $html = ts3viewer_get_data_html();

    // Отправляем результат клиенту
    echo $html;

    // Завершаем выполнение запроса
    wp_die();
}
add_action('wp_ajax_ts3viewer_refresh', 'ts3viewer_ajax_refresh');        // для авторизованных
add_action('wp_ajax_nopriv_ts3viewer_refresh', 'ts3viewer_ajax_refresh'); // для гостей


/* === ШОРТКОД [ts3viewer] === */
function ts3viewer_shortcode() {
    ob_start(); ?>
    <div id="ts3viewer-wrapper">
        <div id="ts3viewer-content">
            <?php echo ts3viewer_get_data_html(); ?>
        </div>
    </div>
    <?php
    return ob_get_clean();
}
add_shortcode('ts3viewer', 'ts3viewer_shortcode');
